import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Languages, TrendingUp, Star, BookOpen, Award, Globe, Target, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const [courses, setCourses] = useState([])

  useEffect(() => {
    loadCourses()
    const interval = setInterval(loadCourses, 500)
    return () => clearInterval(interval)
  }, [])

  const getIconComponent = (iconType) => {
    const icons = {
      'globe': Globe,
      'languages': Languages,
      'book': BookOpen,
      'award': Award
    }
    const IconComponent = icons[iconType] || Globe
    return <IconComponent size={40} strokeWidth={2} />
  }

  const loadCourses = () => {
    const saved = localStorage.getItem('languageCourses')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setCourses(parsed)
      } catch (e) {
        setCourses([])
      }
    } else {
      const defaultCourses = [
        {
          id: 1,
          language: 'Английский',
          level: 'Intermediate',
          progress: 65,
          rating: 5,
          iconType: 'globe'
        },
        {
          id: 2,
          language: 'Испанский',
          level: 'Beginner',
          progress: 30,
          rating: 4,
          iconType: 'languages'
        },
        {
          id: 3,
          language: 'Китайский',
          level: 'Advanced',
          progress: 85,
          rating: 5,
          iconType: 'book'
        }
      ]
      setCourses(defaultCourses)
      localStorage.setItem('languageCourses', JSON.stringify(defaultCourses))
    }
  }

  const getLanguageColor = (language) => {
    const colors = {
      'Английский': '#3B82F6',
      'Испанский': '#EF4444',
      'Китайский': '#F59E0B',
      'Немецкий': '#10B981'
    }
    return colors[language] || '#6366F1'
  }

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={16}
        fill={i < rating ? '#F59E0B' : 'none'}
        stroke={i < rating ? '#F59E0B' : '#CBD5E1'}
        strokeWidth={1.5}
      />
    ))
  }

  const averageRating = courses.length > 0 
    ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1)
    : 0

  const totalProgress = courses.length > 0
    ? Math.round(courses.reduce((sum, c) => sum + c.progress, 0) / courses.length)
    : 0

  return (
    <div className="my-courses-dashboard">
      <div className="dashboard-layout">
        <aside className="dashboard-sidebar">
          <motion.div 
            className="sidebar-header"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.5 }}
          >
            <div className="sidebar-logo">
              <Languages size={32} strokeWidth={2.5} className="sidebar-logo-icon" />
              <h2 className="sidebar-logo-text">LangMaster</h2>
            </div>
            <p className="sidebar-subtitle">Панель управления</p>
          </motion.div>

          <motion.div 
            className="sidebar-metrics"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.5, delay: 0.2 }}
          >
            <div className="metric-item">
              <div className="metric-icon-wrapper" style={{ background: 'rgba(99, 102, 241, 0.15)' }}>
                <Target size={24} strokeWidth={2} color="#6366F1" />
              </div>
              <div className="metric-info">
                <div className="metric-value">{courses.length}</div>
                <div className="metric-label">Курсов</div>
              </div>
            </div>
            <div className="metric-item">
              <div className="metric-icon-wrapper" style={{ background: 'rgba(16, 185, 129, 0.15)' }}>
                <TrendingUp size={24} strokeWidth={2} color="#10B981" />
              </div>
              <div className="metric-info">
                <div className="metric-value">{totalProgress}%</div>
                <div className="metric-label">Прогресс</div>
              </div>
            </div>
            <div className="metric-item">
              <div className="metric-icon-wrapper" style={{ background: 'rgba(245, 158, 11, 0.15)' }}>
                <Award size={24} strokeWidth={2} color="#F59E0B" />
              </div>
              <div className="metric-info">
                <div className="metric-value">{averageRating}</div>
                <div className="metric-label">Рейтинг</div>
              </div>
            </div>
          </motion.div>
        </aside>

        <main className="dashboard-main">
          <motion.div 
            className="dashboard-header"
            initial={{ opacity: 0, y: -20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.5 }}
          >
            <div>
              <h1 className="dashboard-title">Мои курсы</h1>
              <p className="dashboard-subtitle">Управляйте своими языковыми курсами</p>
            </div>
            <div className="header-actions">
              <Clock size={20} strokeWidth={2} />
              <span>{new Date().toLocaleDateString('ru-RU', { day: 'numeric', month: 'long' })}</span>
            </div>
          </motion.div>

          <ImageSlider images={[
            '/images/language-1.jpg',
            '/images/language-2.jpg',
            '/images/language-3.jpg',
            '/images/language-4.jpg'
          ]} />

          <motion.section 
            className="courses-masonry"
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ duration: 0.6, delay: 0.3 }}
          >
            {courses.length > 0 ? (
              <div className="masonry-grid">
                {courses.map((course, index) => (
                  <motion.div 
                    key={course.id} 
                    className="masonry-card"
                    initial={{ opacity: 0, scale: 0.9 }}
                    animate={{ opacity: 1, scale: 1 }}
                    transition={{ delay: index * 0.15, duration: 0.4 }}
                    whileHover={{ rotate: 1, boxShadow: '0 20px 40px rgba(99, 102, 241, 0.25)' }}
                    style={{ 
                      borderLeft: `4px solid ${getLanguageColor(course.language)}`,
                      background: `linear-gradient(135deg, rgba(255, 255, 255, 0.98) 0%, rgba(255, 255, 255, 0.95) 100%)`
                    }}
                  >
                    <div className="masonry-card-top">
                      <div className="masonry-icon-circle" style={{ background: `${getLanguageColor(course.language)}20` }}>
                        {getIconComponent(course.iconType || 'globe')}
                      </div>
                      <div className="masonry-badge" style={{ background: `${getLanguageColor(course.language)}15`, color: getLanguageColor(course.language) }}>
                        {course.level}
                      </div>
                    </div>
                    <div className="masonry-card-content">
                      <h3 className="masonry-language">{course.language}</h3>
                      <div className="masonry-progress-container">
                        <div className="masonry-progress-info">
                          <span className="masonry-progress-label">Прогресс обучения</span>
                          <span className="masonry-progress-percent">{course.progress}%</span>
                        </div>
                        <div className="masonry-progress-track">
                          <motion.div 
                            className="masonry-progress-fill"
                            initial={{ width: 0 }}
                            animate={{ width: `${course.progress}%` }}
                            transition={{ duration: 1, delay: index * 0.15 + 0.3 }}
                            style={{ background: getLanguageColor(course.language) }}
                          />
                        </div>
                      </div>
                      <div className="masonry-rating-section">
                        <div className="masonry-stars">
                          {renderStars(course.rating)}
                        </div>
                        <span className="masonry-rating-text">Оценка: {course.rating}/5</span>
                      </div>
                    </div>
                  </motion.div>
                ))}
              </div>
            ) : (
              <div className="empty-dashboard">
                <div className="empty-icon-circle">
                  <BookOpen size={48} strokeWidth={1.5} />
                </div>
                <h3 className="empty-title">Нет активных курсов</h3>
                <p className="empty-text">Начните изучение нового языка прямо сейчас</p>
              </div>
            )}
          </motion.section>
        </main>
      </div>
    </div>
  )
}

export default MyCourses
